#ifndef __TAudioBuffer__
#define __TAudioBuffer__

//	===========================================================================

#include <Basics/CCountedObject.hpp>
#include <MathTools/CMathTools.hpp>
#include <Collections/TAutomaticPointer.hpp>
using Exponent::MathTools::CMathTools;
using Exponent::Basics::CCountedObject;
using Exponent::Collections::TAutomaticPointer;

//	===========================================================================

namespace Exponent
{
	namespace Audio
	{
		/**
		 * @class TAudioBuffer TAudioBuffer.hpp
		 * @brief Template'd audio data collection
		 *
		 * @date 07/04/2006
		 * @author Paul Chana
		 * @version 1.0.0 Initial version
		 *
		 * @note All contents of this source code are copyright 2005 Exp Digital Uk.\n
		 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy\n
		 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
		 * All content is the Intellectual property of Exp Digital Uk.\n
		 * Certain sections of this code may come from other sources. They are credited where applicable.\n
		 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
		 *
		 * $Id: TAudioBuffer.hpp,v 1.9 2007/02/08 21:08:09 paul Exp $
		 */
		template<typename TypeName> class TAudioBuffer : public CCountedObject
		{
			/** @cond */
			EXPONENT_CLASS_DECLARATION;
			/** @endcond */

//	===========================================================================

		public:

//	===========================================================================

			/**
			* @typedef TAudioBufferPointer
			* @brief Typdefed Automatic pointer to an audio buffer
			*/
			typedef TAutomaticPointer< TAutomaticPointer<TypeName> > TAudioBufferPointer;

//	===========================================================================

			/**
			 * Construction
			 */
			TAudioBuffer() : m_buffer(NULL), m_numberOfSamples(0), m_numberOfChannels(0), m_bufferSize(0)
			{
				NULL_POINTER(m_buffer);
				m_numberOfSamples  = 0;
				m_numberOfChannels = 0;
				m_bufferSize	   = 0;
			}

			/**
			 * Destruction
			 */
			virtual ~TAudioBuffer()
			{
				this->uninitialise();
			}

//	===========================================================================


			/**
			 * Initialise with a specific number of samples and channels. Implies clearBuffer()
			 * @param numberOfSamples The length of each channel in samples
			 * @param numberOfChannels The number of channels in the audio buffer
			 * @retval bool True if initialised properly, false otherwise
			 */
			bool initialise(const unsigned long numberOfSamples, const unsigned long numberOfChannels)
			{
				// Delete the old buffer
				if (m_buffer)
				{
					this->uninitialise();
				}

				// Total size of the buffer
				m_numberOfSamples  = numberOfSamples;
				m_numberOfChannels = numberOfChannels;
				m_bufferSize	   = m_numberOfSamples * m_numberOfChannels;

				// Create the buffer
				m_buffer = new TypeName[m_bufferSize];

				// Clear the audio buffers out
				this->clearBuffer();

				// We are done :)
				return true;
			}

			/**
			 * Clear and delete all buffers
			 */
			void uninitialise()
			{
				// Check we've got a buffer
				if (m_buffer)
				{
					// Delete the main buffer
					FREE_ARRAY_POINTER(m_buffer);

					// Default values
					m_numberOfChannels = 0;
					m_numberOfSamples  = 0;
					m_bufferSize	   = 0;
				}
			}

			/**
			 * Clear the audio data, sets all to zero
			 */
			void clearBuffer()
			{
				// Check we've got a buffer
				if (m_buffer)
				{
					memset(m_buffer, 0, m_bufferSize * sizeof(TypeName));
				}
			}

//	===========================================================================

			/**
			 * Get a channel
			 * @param index The channel required
			 * @retval const TypeName* The buffer or NULL on error
			 * @see getChannel
			 */
			const TypeName *operator[](const long index) const
			{
				return this->getChannel(index);
			}

//	===========================================================================

			/**
			 * Get a specific channel
			 * @param index The channel required
			 * @retval const TypeName* A pointer to the starting sample of this channel\n
			 * note that this may not necessarilly be the first sample in the actual audio\n
			 * data (due to interleaving) and you should not increment automatically by 1\n
			 * as the frame size may be bigger
			 * @see getSampleFrameSize
			 */
			const TypeName *getChannel(const unsigned long index) const
			{
				if (m_buffer && index < m_numberOfChannels)
				{
					return &m_buffer[index];
				}
				return NULL;
			}

			/**
			 * Get a specific channel
			 * @param index The channel required
			 * @retval TypeName* A pointer to the starting sample of this channel\n
			 * note that this may not necessarilly be the first sample in the actual audio\n
			 * data (due to interleaving) and you should not increment automatically by 1\n
			 * as the frame size may be bigger
			 * @see getSampleFrameSize
			 */
			TypeName *getMutableChannel(const unsigned long index)
			{
				if (m_buffer && index < m_numberOfChannels)
				{
					return &m_buffer[index];
				}
				return NULL;
			}

			/**
			 * Get a pointer to the start of the data
			 * @retval const TypeName* A pointer to the data
			 */
			const TypeName *getData() const
			{
				if (m_buffer)
				{
					return m_buffer;
				}
				return NULL;
			}

			/**
			 * Get a pointer to the start of the data
			 * @retval TypeName* A pointer to the data
			 */
			TypeName *getMutableData()
			{
				if (m_buffer)
				{
					return m_buffer;
				}
				return NULL;
			}

//	===========================================================================

			/**
			 * Get the number of audio channels
			 * @retval unsigned long The number of channels
			 */
			FORCEINLINE unsigned long getNumberOfChannels() const { return m_numberOfChannels; }

			/**
			 * Get the number of audio samples per channel
			 * @retval unsigned long The number of samples per channel
			 */
			FORCEINLINE unsigned long getNumberOfSamples() const { return m_numberOfSamples; }

			/**
			 * Get teh size of the buffer
			 * @retval unsigned long The total size of the buffer
			 */
			FORCEINLINE unsigned long getBufferSize() const { return m_bufferSize; }

			/**
			 * Each block of audio samples is interleaved with the other channels samples in to a frame\n
			 * When processing you need to increment each channel point by a certain amount to move to its next sample\n
			 * That value is the sample frame size returned here
			 * @retval unsigned long The size of a sample frame
			 */
			FORCEINLINE unsigned long getSampleFrameSize() const { return m_numberOfChannels; }

//	===========================================================================

			/**
			 * Get the maximum sample in the entre buffer
			 * @param channel The channel to check
			 * @retval TypeName The maximum sample
			 */
			TypeName getMaximumSample(const unsigned long channel) const
			{
				if (channel < m_numberOfChannels && m_buffer)
				{
					TypeName *buffer  = &m_buffer[channel];
					TypeName maxValue = -2.0;	// Well outside of range
					for (long i = 0; i < m_numberOfSamples; i++)
					{
						maxValue = CMathTools::fastMaximum(*buffer, maxValue);
						buffer  += m_numberOfChannels;
					}
					return maxValue;
				}
				return (TypeName)0;
			}

			/**
			 * Get the maximum sample in the entre buffer
			 * @param channel The channel to check if -1 all channels are check
			 * @retval TypeName The maximum sample
			 */
			TypeName getMinimumSample(const unsigned long channel) const
			{
				if (channel < m_numberOfChannels && m_buffer)
				{
					TypeName *buffer  = &m_buffer[channel];
					TypeName minValue = 2.0;	// Well outside of range
					for (long i = 0; i < m_numberOfSamples; i++)
					{
						minValue = CMathTools::fastMinimum(*buffer, minValue);
						buffer  += m_numberOfChannels;
					}
					return minValue;
				}
				return (TypeName)0;
			}

			/**
			 * Get the RMS level of a particular channel
			 * @param channel The channel to check if -1 all channels are check
			 * @param numberOfSamples The number of samples to process
			 * @param startSample The sample to start on
			 * @retval TypeName The RMS value of the buffer in the range  0 - 1
			 */
			FORCEINLINE TypeName getRMSLevel(const unsigned long channel, const unsigned long numberOfSamples, const unsigned long startSample = 0) const
			{
				if (channel < m_numberOfChannels && m_buffer && numberOfSamples <= m_numberOfSamples && startSample + numberOfSamples <= m_numberOfSamples)
				{
					TypeName mean = 0.0;
					const TypeName *theChannel = &m_buffer[channel + (this->getSampleFrameSize() * startSample)];

					// Compute the squares
					for (unsigned long i = 0; i < numberOfSamples; i++)
					{
						mean += *theChannel * *theChannel;
						theChannel += this->getSampleFrameSize();
					}

					// Return the RMS
					return sqrt(mean / (TypeName)numberOfSamples);
				}
				return (TypeName)0;
			}

			/**
			 * Denormal the audio data
			 */
			void denormal()
			{
				// Check the buffer is valid
				if (m_buffer == NULL)
				{
					return;
				}

				// Loop and apply the denormalling..
				for (unsigned long i = 0; i < m_bufferSize; i++)
				{
					m_buffer[i] += (TypeName)1e-15;
				}
			}

			/**
			 * Merge all the channels to mono at a level commensurate to the number of channels
			 * @retval bool True if sucessfully merged or mono channel, false on errors
			 */
			bool mergeChannelsToMono()
			{
				// Check we have an audio buffer
				if (m_buffer == NULL)
				{
					return false;
				}

				// Check if we are mono already
				if (m_numberOfChannels <= 1)
				{
					return true;		// We it is a sucess really ;)
				}

				// Compute the level of each channel
				const TypeName level = 1.0 / (TypeName)m_numberOfChannels;

				// Store the internal buffer
				TypeName *buffer = new TypeName[m_numberOfSamples];				// Note single channel!!
				memset(buffer, 0, m_numberOfSamples * sizeof(TypeName));
				TypeName *bufferPointer = &m_buffer[0];

				// Merge the samples
				for (unsigned long i = 0; i < m_numberOfSamples; i++)
				{
					for (unsigned long j = 0; j < m_numberOfChannels; j++)
					{
						buffer[i] += *bufferPointer++ * level;
					}
				}

				// Delete the old buffer
				FREE_POINTER(m_buffer);

				// Store the new buffer
				m_buffer = buffer;

				// Set the number of channels
				m_numberOfChannels = 1;
				m_bufferSize	   = m_numberOfSamples;

				// We are complete!
				return true;
			}

			/**
			 * Copy a certain number of samples from a buffer
			 * @param buffer The buffer to copy. Assumed to be in correct interleaved format
			 * @param numberOfSamples The total number of samples (must be <= m_numberOfSamples * m_numberOfChannels)
			 * @retval bool True if copied properly, false otherwise
			 */
			bool copySamples(const TypeName *buffer, const unsigned long numberOfSamples)
			{
				// Check if we have room
				if (numberOfSamples <= m_numberOfSamples * m_numberOfChannels && m_buffer)
				{
					// Copy the buffer
					memcpy(m_buffer, buffer, numberOfSamples * sizeof(TypeName));

					// Complete!
					return true;
				}

				// Failed
				return false;
			}

			/**
			 * Merge a channel to the passed audio buffer
			 * @param buffer The output audio buffer
			 * @param startSample The starting sample in this buffer
			 * @param numberOfSamples The number of samples to process
			 * @param channel The audio channel to merge
			 * @retval bool True if happened properly, false otherwise
			 * @note startSample + numberOfSamples must be <= m_numberOfSamples
			 */
			bool mergeChannelToBuffer(float *buffer, const unsigned long startSample, const unsigned long numberOfSamples, const unsigned long channel)
			{
				// Compute the length to process
				const unsigned long length = startSample + numberOfSamples;

				// Check for the validity
				if (length <= m_numberOfSamples && channel < m_numberOfChannels && buffer != NULL)
				{
					//&m_buffer[index];

					// Get the audio data
					const TypeName *audioData = &m_buffer[channel][startSample * m_numberOfChannels];

					// Check we got proper audio data
					if (audioData == NULL)
					{
						return false;
					}

					// Now loop through and process
					for (unsigned long i = 0; i < numberOfSamples; i++)
					{
						buffer[i] += (float)*audioData;
						audioData += m_numberOfChannels;
					}

					// Sucess
					return true;
				}

				// Failed
				return false;
			}

			/**
			 * Merge a channel to the passed audio buffer
			 * @param buffer The output audio buffer
			 * @param startSample The starting sample in this buffer
			 * @param numberOfSamples The number of samples to process
			 * @param channel The audio channel to merge
			 * @retval bool True if happened properly, false otherwise
			 * @note startSample + numberOfSamples must be <= m_numberOfSamples
			 */
			bool mergeChannelToBuffer(double *buffer, const unsigned long startSample, const unsigned long numberOfSamples, const unsigned long channel)
			{
				// Compute the length to process
				const unsigned long length = startSample + numberOfSamples;

				// Check for the validity
				if (length <= m_numberOfSamples && channel < m_numberOfChannels && buffer != NULL)
				{
					//&m_buffer[index];

					// Get the audio data
					const TypeName *audioData = &m_buffer[startSample * m_numberOfChannels];

					// Now loop through and process
					for (unsigned long i = 0; i < numberOfSamples; i++)
					{
						buffer[i] += (double)*audioData;
						audioData += m_numberOfChannels;
					}

					// Sucess
					return true;
				}

				// Failed
				return false;
			}

//	===========================================================================

		protected:

//	===========================================================================

			TypeName *m_buffer;					/**< The actual audio data */
			unsigned long m_numberOfSamples;	/**< The number of samples per channel */
			unsigned long m_numberOfChannels;	/**< The number of channels */
			unsigned long m_bufferSize;			/**< Size of the buffer */

//	===========================================================================

		};

		/** 
		 * @typedef TDoublePrecisionAudioBuffer
		 * @brief 64 Bit double precision audio buffer
		 */
		typedef TAudioBuffer<double> TDoublePrecisionAudioBuffer;

		/** 
		 * @typedef TSinglePrecisionAudioBuffer
		 * @brief 32 Bit float precision audio buffer
		 */
		typedef TAudioBuffer<float> TSinglePrecisionAudioBuffer;

		/** @cond */
		 EXPONENT_TEMPLATE_CLASS_IMPLEMENTATION(TAudioBuffer<TypeName>, TypeName, CCountedObject);
		 /** @endcond */
	}
}
#endif	// End of TAudioBuffer.hpp